using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using BLOODUNITFINANCE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitFinance;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using BLOODUNITSHIPMENT = gov.va.med.vbecs.Common.VbecsTables.BloodUnitShipment;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Luke Meyer</Developer>
	///	<Developer>Krzysztof Dobranowski</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/16/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// This class implements data persistence logic for blood unit finance data. 
	/// It exposes set of public static methods retrieving from / storing data to the VBECS database.
	///</summary>

	#endregion

	public class BloodUnitFinance
	{
		

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/10/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7493"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7494"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// spInsertBloodUnitFinance, spUpdateBloodUnitFinance
		/// </summary>
		/// <param name="isUpdate"></param>
		/// <returns></returns>
		public static DataTable GetEmptyBloodUnitFinanceTableSchema(bool isUpdate)
		{
			DataTable dtBloodUnitFinance = new DataTable(BLOODUNITFINANCE.TableName);

			dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.BloodUnitFinanceGuid, typeof(System.Guid));
			dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.UnitCost, typeof(decimal));
			dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.SpecialTestingCost, typeof(decimal));
			dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.ReturnCreditAmount, typeof(decimal));

			if (isUpdate)
			{
				dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.RowVersion, typeof(byte[]));
			}
			else
			{
				dtBloodUnitFinance.Columns.Add(BLOODUNITFINANCE.BloodUnitGuid, typeof(System.Guid));
			}

			return(dtBloodUnitFinance);
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/16/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="664"> 
		///		<ExpectedInput>Eye readable unit identifier, product code and division code</ExpectedInput>
		///		<ExpectedOutput>The data row which contains financial data about a blood unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1894"> 
		///		<ExpectedInput>Empty strings</ExpectedInput>
		///		<ExpectedOutput>RowNotInTableException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves blood unit financial data from a database based on blood unit ID 
		/// and/or product code and division code.
		/// One of two parameters: blood unit ID and product code may be blank. 
		/// Division code is mandatory parameter. 
		/// </summary>
		/// <param name="unitId">Eyereadable Blood unit ID - either an eye readable unit ID or label unit ID. Optional - may be blank.</param>
		/// <param name="productCode">Product code. Optional - may be blank.</param>
		/// <returns>DataRow containing blood unit financial data</returns>		
		public static DataRow GetBloodUnitFinancialData( string unitId, string productCode)
		{
			SqlParameter[] prms = {
									  new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.divisioncode, Common.LogonUser.LogonUserDivisionCode),
									  new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.unitid, null),
									  new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.productcode, null)
								  };
			
			bool bHasUnitID = false;

			if( unitId != String.Empty)
			{
				bHasUnitID = true;
				prms[1] = new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.unitid, unitId);
			}
		
			if(productCode != String.Empty)
			{

				if(bHasUnitID)
				{
					prms[2] = new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.productcode, productCode);
				}
				else
				{
					prms[1] = new SqlParameter(STOREDPROC.GetBloodUnitFinancialData.productcode, productCode);
				}
			}
		
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitFinancialData.StoredProcName, prms);
		
			switch (ds.Tables[0].Rows.Count)
			{
				case 0:
					//No row was found for a blood unit from this division.
					return ds.Tables[0].NewRow();
				case 1:
					return ds.Tables[0].Rows[0];
				default:
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.TooManyRecordsFoundException( 
										  Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
					throw new DataAccessLayerException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="846"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="847"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IsBloodUnitOnPendingOutgoingShipmentList
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool IsBloodUnitOnPendingOutgoingShipmentList(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsBloodUnitOnPendingOutgoingShipmentList.BloodUnitGuid, System.Data.SqlDbType.UniqueIdentifier)
			};
			
			prms[0].Value = bloodUnitGuid;
           
			return new StoredProcedure().GetBoolReturnValue( STOREDPROC.IsBloodUnitOnPendingOutgoingShipmentList.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="848"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of BloodUnits</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="854"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetFinancialBloodUnits
		/// </summary>
		/// <returns></returns>
		public static DataTable GetFinancialBloodUnits()
		{
			SqlParameter[] _prms =
			{
				new SqlParameter(STOREDPROC.GetFinancialBloodUnits.divisioncode, System.Data.SqlDbType.Char)
			};

			_prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			return Common.StoredProcedure.GetData(STOREDPROC.GetFinancialBloodUnits.StoredProcName, _prms).Tables[0];
		}
        

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="855"> 
		///		<ExpectedInput>BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Finance information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="863"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetBloodUnitFinanceByUnitGuid
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitFinanceByUnitGuid(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetBloodUnitFinanceByUnitGuid.bloodunitguid,System.Data.SqlDbType.UniqueIdentifier),
					//CR 3195
					new SqlParameter(STOREDPROC.GetBloodUnitFinanceByUnitGuid.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = bloodUnitGuid;	
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetBloodUnitFinanceByUnitGuid.StoredProcName, prms).Tables[0];
		}
        

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/13/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="873"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="874"> 
		///		<ExpectedInput>InvalidDataRow</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Save
		/// </summary>
		/// <param name="bloodUnitFinanceDataRow"></param>
		/// <returns></returns>
		public static bool Save(DataRow bloodUnitFinanceDataRow)
		{
			int retValue = new Common.StoredProcedure().TransactionalGetValue(STOREDPROC.UpdateBloodUnitFinance.StoredProcName, bloodUnitFinanceDataRow);
			return (retValue == 0);
		}

	}		
}



